import { NextResponse } from 'next/server'
import { getServerSession } from 'next-auth'
import { authOptions } from '@/lib/auth'
import { prisma } from '@/lib/prisma'

export async function GET() {
  try {
    const session = await getServerSession(authOptions)
    if (!session) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    const contents = await prisma.content.findMany({
      where: {
        key: {
          in: ['about_text', 'vision', 'mission', 'values']
        }
      }
    })

    const contentMap: any = {
      aboutAr: '',
      aboutEn: '',
      visionAr: '',
      visionEn: '',
      missionAr: '',
      missionEn: '',
      valuesAr: '',
      valuesEn: '',
    }

    contents.forEach((item: any) => {
      if (item.key === 'about_text') {
        contentMap.aboutAr = item.valueAr || ''
        contentMap.aboutEn = item.valueEn || ''
      }
      if (item.key === 'vision') {
        contentMap.visionAr = item.valueAr || ''
        contentMap.visionEn = item.valueEn || ''
      }
      if (item.key === 'mission') {
        contentMap.missionAr = item.valueAr || ''
        contentMap.missionEn = item.valueEn || ''
      }
      if (item.key === 'values') {
        contentMap.valuesAr = item.valueAr || ''
        contentMap.valuesEn = item.valueEn || ''
      }
    })

    return NextResponse.json(contentMap)
  } catch (error) {
    console.error(error)
    return NextResponse.json({ error: 'Failed to fetch content' }, { status: 500 })
  }
}

export async function POST(request: Request) {
  try {
    const session = await getServerSession(authOptions)
    if (!session) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    const body = await request.json()

    await prisma.content.upsert({
      where: { key: 'about_text' },
      update: { valueAr: body.aboutAr, valueEn: body.aboutEn },
      create: { key: 'about_text', valueAr: body.aboutAr, valueEn: body.aboutEn }
    })

    await prisma.content.upsert({
      where: { key: 'vision' },
      update: { valueAr: body.visionAr, valueEn: body.visionEn },
      create: { key: 'vision', valueAr: body.visionAr, valueEn: body.visionEn }
    })

    await prisma.content.upsert({
      where: { key: 'mission' },
      update: { valueAr: body.missionAr, valueEn: body.missionEn },
      create: { key: 'mission', valueAr: body.missionAr, valueEn: body.missionEn }
    })

    await prisma.content.upsert({
      where: { key: 'values' },
      update: { valueAr: body.valuesAr, valueEn: body.valuesEn },
      create: { key: 'values', valueAr: body.valuesAr, valueEn: body.valuesEn }
    })

    await prisma.content.upsert({
      where: { key: 'values_image' },
      update: { valueAr: body.valuesImage, valueEn: body.valuesImage },
      create: { key: 'values_image', valueAr: body.valuesImage, valueEn: body.valuesImage }
    })

    return NextResponse.json({ success: true })
  } catch (error) {
    console.error(error)
    return NextResponse.json({ error: 'Failed to save content' }, { status: 500 })
  }
}
