import { NextResponse } from 'next/server'
import { getServerSession } from 'next-auth'
import { authOptions } from '@/lib/auth'
import { prisma } from '@/lib/prisma'

export async function GET() {
  try {
    const session = await getServerSession(authOptions)
    if (!session) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    const accreditations = await prisma.accreditation.findMany({
      orderBy: { order: 'asc' }
    })

    // Map logo to imageUrl for frontend
    const mapped = accreditations.map((item: any) => ({
      ...item,
      imageUrl: item.logo,
      descriptionAr: '',
      descriptionEn: ''
    }))

    return NextResponse.json(mapped)
  } catch (error) {
    return NextResponse.json({ error: 'Failed to fetch' }, { status: 500 })
  }
}

export async function POST(request: Request) {
  try {
    const session = await getServerSession(authOptions)
    if (!session) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    const body = await request.json()

    if (body.isNew) {
      await prisma.accreditation.create({
        data: {
          nameAr: body.nameAr,
          nameEn: body.nameEn,
          logo: body.imageUrl || '',
          order: body.order || 0,
        }
      })
    } else {
      await prisma.accreditation.update({
        where: { id: body.id },
        data: {
          nameAr: body.nameAr,
          nameEn: body.nameEn,
          logo: body.imageUrl || '',
        }
      })
    }

    return NextResponse.json({ success: true })
  } catch (error) {
    return NextResponse.json({ error: 'Failed to save' }, { status: 500 })
  }
}

export async function DELETE(request: Request) {
  try {
    const session = await getServerSession(authOptions)
    if (!session) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    const { searchParams } = new URL(request.url)
    const id = searchParams.get('id')

    if (!id) {
      return NextResponse.json({ error: 'ID required' }, { status: 400 })
    }

    await prisma.accreditation.delete({ where: { id } })
    return NextResponse.json({ success: true })
  } catch (error) {
    return NextResponse.json({ error: 'Failed to delete' }, { status: 500 })
  }
}
