import { NextResponse } from 'next/server'
import { getServerSession } from 'next-auth'
import { authOptions } from '@/lib/auth'
import { prisma } from '@/lib/prisma'

export async function GET() {
  try {
    const session = await getServerSession(authOptions)
    if (!session) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    const contents = await prisma.content.findMany({
      where: {
        key: { in: ['transparency', 'commitment', 'innovation'] }
      }
    })

    const contentMap: any = {
      transparencyAr: '',
      transparencyEn: '',
      commitmentAr: '',
      commitmentEn: '',
      innovationAr: '',
      innovationEn: '',
    }

    contents.forEach((item: any) => {
      if (item.key === 'transparency') {
        contentMap.transparencyAr = item.valueAr || ''
        contentMap.transparencyEn = item.valueEn || ''
      }
      if (item.key === 'commitment') {
        contentMap.commitmentAr = item.valueAr || ''
        contentMap.commitmentEn = item.valueEn || ''
      }
      if (item.key === 'innovation') {
        contentMap.innovationAr = item.valueAr || ''
        contentMap.innovationEn = item.valueEn || ''
      }
    })

    return NextResponse.json(contentMap)
  } catch (error) {
    return NextResponse.json({ error: 'Failed to fetch' }, { status: 500 })
  }
}

export async function POST(request: Request) {
  try {
    const session = await getServerSession(authOptions)
    if (!session) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    const body = await request.json()

    await prisma.content.upsert({
      where: { key: 'transparency' },
      update: { valueAr: body.transparencyAr, valueEn: body.transparencyEn },
      create: { key: 'transparency', valueAr: body.transparencyAr, valueEn: body.transparencyEn }
    })

    await prisma.content.upsert({
      where: { key: 'commitment' },
      update: { valueAr: body.commitmentAr, valueEn: body.commitmentEn },
      create: { key: 'commitment', valueAr: body.commitmentAr, valueEn: body.commitmentEn }
    })

    await prisma.content.upsert({
      where: { key: 'innovation' },
      update: { valueAr: body.innovationAr, valueEn: body.innovationEn },
      create: { key: 'innovation', valueAr: body.innovationAr, valueEn: body.innovationEn }
    })

    return NextResponse.json({ success: true })
  } catch (error) {
    return NextResponse.json({ error: 'Failed to save' }, { status: 500 })
  }
}
