import { NextResponse } from 'next/server'
import { getServerSession } from 'next-auth'
import { authOptions } from '@/lib/auth'
import { prisma } from '@/lib/prisma'

export async function GET() {
  try {
    const session = await getServerSession(authOptions)
    if (!session) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    const team = await prisma.teamMember.findMany({
      orderBy: { order: 'asc' }
    })

    // Map image to imageUrl for frontend
    const mappedTeam = team.map((member: any) => ({
      ...member,
      imageUrl: member.image
    }))

    return NextResponse.json(mappedTeam)
  } catch (error) {
    return NextResponse.json({ error: 'Failed to fetch' }, { status: 500 })
  }
}

export async function POST(request: Request) {
  try {
    const session = await getServerSession(authOptions)
    if (!session) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    const body = await request.json()

    if (body.isNew) {
      await prisma.teamMember.create({
        data: {
          nameAr: body.nameAr,
          nameEn: body.nameEn,
          positionAr: body.positionAr,
          positionEn: body.positionEn,
          image: body.imageUrl || '',
          order: body.order || 0,
        }
      })
    } else {
      await prisma.teamMember.update({
        where: { id: body.id },
        data: {
          nameAr: body.nameAr,
          nameEn: body.nameEn,
          positionAr: body.positionAr,
          positionEn: body.positionEn,
          image: body.imageUrl || '',
        }
      })
    }

    return NextResponse.json({ success: true })
  } catch (error) {
    return NextResponse.json({ error: 'Failed to save' }, { status: 500 })
  }
}

export async function DELETE(request: Request) {
  try {
    const session = await getServerSession(authOptions)
    if (!session) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    const { searchParams } = new URL(request.url)
    const id = searchParams.get('id')

    if (!id) {
      return NextResponse.json({ error: 'ID required' }, { status: 400 })
    }

    await prisma.teamMember.delete({ where: { id } })
    return NextResponse.json({ success: true })
  } catch (error) {
    return NextResponse.json({ error: 'Failed to delete' }, { status: 500 })
  }
}
