'use client'

import { useTranslations } from 'next-intl'
import { ChevronLeft, ChevronRight } from 'lucide-react'
import { useState, useEffect } from 'react'
import Image from 'next/image'

export default function ClientsSection() {
  const t = useTranslations('clients')
  const [currentIndex, setCurrentIndex] = useState(0)
  const [clients, setClients] = useState<any[]>([])
  const itemsPerPage = 5

  useEffect(() => {
    fetch('/api/clients')
      .then(res => res.json())
      .then(data => setClients(data))
      .catch(err => console.error(err))
  }, [])

  const next = () => {
    if (clients.length > itemsPerPage) {
      setCurrentIndex((prev) => (prev + 1) % (clients.length - itemsPerPage + 1))
    }
  }

  const prev = () => {
    if (clients.length > itemsPerPage) {
      setCurrentIndex((prev) => (prev - 1 + (clients.length - itemsPerPage + 1)) % (clients.length - itemsPerPage + 1))
    }
  }

  useEffect(() => {
    if (clients.length > itemsPerPage) {
      const interval = setInterval(() => {
        next()
      }, 3000)
      return () => clearInterval(interval)
    }
  }, [currentIndex, clients.length])

  return (
    <section className="py-20 bg-white">
      <div className="container mx-auto px-4">
        <h2 className="text-4xl md:text-5xl font-bold text-center mb-4 text-[#33335E]">
          {t('title')}
        </h2>
        <p className="text-center text-gray-600 mb-12 max-w-3xl mx-auto text-sm">
          {t('subtitle')}
        </p>

        <div className="relative max-w-6xl mx-auto">
          <button
            onClick={prev}
            className="absolute left-0 top-1/2 -translate-y-1/2 z-10 flex items-center justify-center hover:opacity-80 transition-opacity"
          >
            <ChevronLeft className="w-8 h-8 text-[#33335E]" />
          </button>

          <div className="overflow-hidden px-12">
            <div 
              className="flex gap-6 transition-transform duration-500 ease-in-out"
              style={{ transform: `translateX(-${currentIndex * (100 / itemsPerPage)}%)` }}
            >
              {clients.map((client) => (
                <div
                  key={client.id}
                  className="flex-shrink-0 w-44 h-28 bg-white border-2 border-[#33335E] rounded-full flex items-center justify-center hover:shadow-lg transition-shadow px-6"
                  style={{ width: `calc((100% - ${(itemsPerPage - 1) * 1.5}rem) / ${itemsPerPage})` }}
                >
                  {client.logo ? (
                    <div className="relative w-32 h-20">
                      <Image src={client.logo} alt={client.name} fill className="object-contain" />
                    </div>
                  ) : (
                    <div className="text-gray-400 text-center text-xs">
                      {client.name}
                    </div>
                  )}
                </div>
              ))}
            </div>
          </div>

          <button
            onClick={next}
            className="absolute right-0 top-1/2 -translate-y-1/2 z-10 flex items-center justify-center hover:opacity-80 transition-opacity"
          >
            <ChevronRight className="w-8 h-8 text-[#33335E]" />
          </button>
        </div>
      </div>
    </section>
  )
}
