'use client'

import { useState } from 'react'
import { useTranslations } from 'next-intl'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Textarea } from '@/components/ui/textarea'
import { Label } from '@/components/ui/label'
import { Card } from '@/components/ui/card'

export default function ContactSection() {
  const t = useTranslations('contact')
  const [loading, setLoading] = useState(false)
  const [success, setSuccess] = useState(false)

  const handleSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault()
    setLoading(true)

    const formData = new FormData(e.currentTarget)
    const data = {
      name: formData.get('name'),
      phone: formData.get('phone'),
      email: formData.get('email'),
      service: formData.get('service'),
      message: formData.get('message'),
    }

    try {
      const res = await fetch('/api/contact', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(data),
      })

      if (res.ok) {
        setSuccess(true)
        e.currentTarget.reset()
        setTimeout(() => setSuccess(false), 3000)
      }
    } catch (error) {
      console.error(error)
    } finally {
      setLoading(false)
    }
  }

  return (
    <section id="contact" className="py-20 bg-gray-50">
      <div className="container mx-auto px-4">
        <h2 className="text-4xl md:text-5xl font-bold text-center mb-12 text-[#33335E]">
          {t('title')}
        </h2>

        <Card className="max-w-2xl mx-auto p-8">
          <form onSubmit={handleSubmit} className="space-y-6">
            <div>
              <Label htmlFor="name">{t('name')}</Label>
              <Input id="name" name="name" required className="mt-2" />
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label htmlFor="phone">{t('phone')}</Label>
                <Input id="phone" name="phone" type="tel" className="mt-2" />
              </div>
              <div>
                <Label htmlFor="email">{t('email')}</Label>
                <Input id="email" name="email" type="email" className="mt-2" />
              </div>
            </div>

            <div>
              <Label htmlFor="service">{t('service')}</Label>
              <Input id="service" name="service" required className="mt-2" />
            </div>

            <div>
              <Label htmlFor="message">{t('message')}</Label>
              <Textarea id="message" name="message" required rows={5} className="mt-2" />
            </div>

            {success && (
              <div className="p-4 bg-green-50 text-green-700 rounded-lg text-center">
                {t('success')}
              </div>
            )}

            <Button
              type="submit"
              disabled={loading}
              className="w-full bg-[#33335E] hover:bg-[#4a4d7c] text-lg py-6"
            >
              {loading ? '...' : t('send')}
            </Button>
          </form>
        </Card>
      </div>
    </section>
  )
}
